/* ***************************************************************************+
 * ITX package (cnrg.itx) for telephony application programming.              *
 * Copyright (c) 1999  Cornell University, Ithaca NY.                         *
 * A copy of the license is distributed with this package.  Look in the docs  *
 * directory, filename GPL.  Contact information: bergmark@cs.cornell.edu     *
 ******************************************************************************/

package cnrg.itx.gtwy;
import cnrg.itx.datax.*;

/**
 * The user-defined destination of voice data packets from the computer network to the gateway.
 * 
 * @author James Wann
 * @version 1.0a
 * @see PlayChannel
 * @see cnrg.itx.datax.Destination
 */
public class PlayDestination implements Destination
{
	static
	{
		System.loadLibrary("Dialogic");
	}

	/**
	 * The Line instance related to the destination.
	 */
	private Line myLine;
	
	/**
	 * true if destination is open for receiving packets; false otherwise.
	 */
	private boolean isOpen;
	
	/**
	 * true if the destination has to be muted; false otherwise.
	 */
	private boolean isMute;
	
	/**
	 * The number of bytes written to the buffer.
	 */
	private int bytesWritten = 0;
	
	/**
	 * The number of packets written.  Size specified as transfer size to the phone.
	 */
	private int iNumPackets = 0 ;
	
	/**
	 * @param l the <code>Line</code> instance related to <code>PlayDestination</code>.
	 * @see Line
	 */
	public PlayDestination(Line l)
	{
		myLine = l;
		isOpen = true;
		isMute = false;
	}
	
	/**
	 * @return the <code>Line</code> instance for <code>PlayDestination</code>
	 * @see Line
	 */
	public Line getLine()
	{
		return myLine;
	}	
	
	/**
	 * Used to close the destination to receiving voice packets.
	 */
	public void close()
	{
		isOpen = false;
	}
		
	/**
	 * Used to mute the voice data sent to the destination.
	 * 
	 * @param mode true if the destination has to be muted; false otherwise.
	 * @return true if the destination has been muted; false otherwise.
	 */
	public boolean mute(boolean mode)
	{
		isMute = mode;
		return isMute;
	}
		
	/**
	 * Writes voice data to a specified buffer on the gateway.  Data will be played out on a phone later
	 * on.
	 * 
	 * @param data the voice data.
	 */
	public void write(byte[] data) throws DataException
	{
		if (isOpen && !isMute)
		{
			int lineNumber = myLine.getLineNumber();
			
			writeToPlay(lineNumber, data);
			bytesWritten += data.length;
			iNumPackets++;
		}
	}
	
	/**
    * The Gateway is not concerned about properties.  Returns null.
    */ 
   public PropertiesCollection getProperties() throws DataException
   {
	   return null;
   }

   /**
    * No properties are set.
    */
   public void setProperties(PropertiesCollection pc) throws DataException
   {
   }

   /**
    * The Gateway does not set peer properties.
    */
   public void setPeerProperties(PropertiesCollection pc) throws DataException
   {
   }

   /**
	* @return the Stats object which describes the current state of the destination.  
	*/
   public Stats getStatistics()
   {
	   Stats networkStats = new Stats();
	   int lineNumber = myLine.getLineNumber();
		
	   networkStats.addStat("Device", "Gateway Line Destination");
	   networkStats.addStat("<Gateway Line Destination " + lineNumber + "> (Audio) Bytes sent to wire", new Integer(bytesWritten));
	   networkStats.addStat("<Gateway Line Destination " + lineNumber + "> Packets sent to wire ", new Integer(iNumPackets));

	   return networkStats; 
   }
   
   /**
	 * Writes voice data from computer to buffers in native code.  <code>playPacket</code> plays out the
	 * data in the buffers.
	 * 
	 * @param lineNumber the line's designated number.
	 * @param data the data being sent.
	 */
	private native void writeToPlay(int lineNumber, byte[] data);
}